/*
===========================================================================
This file is added for the QUAGENTS 3 project as a place to put our AI nodes.
MAY 26 2011

SEE HEADER FILE.
===========================================================================
*/
//

/*****************************************************************************
 * name:		ai_quagentsnodes.c
 *
 * desc:		Quagents3 bot AI
 *
 *****************************************************************************/

#include "g_local.h"
#include "../botlib/botlib.h"
#include "../botlib/be_aas.h"
#include "../botlib/be_ea.h"
#include "../botlib/be_ai_char.h"
#include "../botlib/be_ai_chat.h"
#include "../botlib/be_ai_gen.h"
#include "../botlib/be_ai_goal.h"
#include "../botlib/be_ai_move.h"
#include "../botlib/be_ai_weap.h"
//
#include "ai_main.h"
#include "ai_dmq3.h"
#include "ai_quagentsnodes.h"
#include "ai_chat.h"
#include "ai_cmd.h"
#include "ai_dmnet.h"
#include "ai_team.h"
//data file headers
#include "chars.h"			//characteristics
#include "inv.h"			//indexes into the inventory
#include "syn.h"			//synonyms
#include "match.h"			//string matching types and vars

// for the voice chats
#include "../../ui/menudef.h"


/*
==================
AI_QuagentsOpCode
==================
*/
void AI_QuagentsOpCode(int commandtype, char* opcode){
	switch(commandtype){
		case QCMD_MI: opcode[0]='m'; opcode[1]='i'; break;
		case QCMD_MB: opcode[0]='m'; opcode[1]='b'; break;
		case QCMD_MT: opcode[0]='m'; opcode[1]='t'; break;
		case QCMD_SK: opcode[0]='s'; opcode[1]='k'; break;
		case QCMD_JU: opcode[0]='j'; opcode[1]='u'; break;
		case QCMD_RO: opcode[0]='r'; opcode[1]='o'; break;
		case QCMD_PO: opcode[0]='p'; opcode[1]='o'; break;
		case QCMD_FA: opcode[0]='f'; opcode[1]='a'; break;
		case QCMD_PA: opcode[0]='p'; opcode[1]='a'; break;
		case QCMD_HC: opcode[0]='h'; opcode[1]='c'; break;
		case QCMD_HM: opcode[0]='h'; opcode[1]='m'; break;
		case QCMD_AC: opcode[0]='a'; opcode[1]='c'; break;
		case QCMD_AM: opcode[0]='a'; opcode[1]='m'; break;
		case QCMD_LC: opcode[0]='l'; opcode[1]='c'; break;
		case QCMD_RF: opcode[0]='r'; opcode[1]='f'; break;
		case QCMD_PK: opcode[0]='p'; opcode[1]='k'; break;
		case QCMD_FW: opcode[0]='f'; opcode[1]='w'; break;
		case QCMD_FC: opcode[0]='f'; opcode[1]='c'; break;
		case QCMD_CS: opcode[0]='c'; opcode[1]='s'; break;
		case QCMD_RA: opcode[0]='r'; opcode[1]='a'; break;
		case QCMD_WI: opcode[0]='w'; opcode[1]='i'; break;
		case QCMD_MF: opcode[0]='m'; opcode[1]='f'; break;
		case QCMD_RB: opcode[0]='r'; opcode[1]='b'; break;
		case QCMD_SW: opcode[0]='s'; opcode[1]='w'; break;
		case QCMD_SC: opcode[0]='s'; opcode[1]='c'; break;
		case QCMD_SV: opcode[0]='s'; opcode[1]='v'; break;
		case QCMD_MC: opcode[0]='m'; opcode[1]='c'; break;
		case QCMD_SY: opcode[0]='s'; opcode[1]='y'; break;
		case QCMD_AL: opcode[0]='a'; opcode[1]='l'; break;
		case QCMD_PU: opcode[0]='p'; opcode[1]='u'; break;
		case QCMD_PD: opcode[0]='p'; opcode[1]='d'; break;
		case QCMD_CI: opcode[0]='c'; opcode[1]='i'; break;
		case QCMD_VC: opcode[0]='v'; opcode[1]='c'; break;
		case QCMD_VD: opcode[0]='v'; opcode[1]='d'; break;
		default: opcode[0]='?'; opcode[1]='?';
	}
	opcode[2]='\0';
}

/*
==================
AIExit_Quagents
==================
*/
void AIExit_Quagents(bot_state_t *bs, int qcommand, char commandtype, char* status){
	char opcode[3];
	char retstring[128];
	AI_QuagentsOpCode(commandtype,opcode);
	Com_sprintf(retstring,127,"cp %s %d %s\n",opcode,qcommand,status);
	if(bs->currentcommand>=bs->finalcommand){
		bs->currentcommand--;
		trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
	}
}

/*
==================
AIEnter_Quagents
==================
*/
void AIEnter_Quagents(bot_state_t *bs, char now, char cmd, unsigned int idnum, char valc1, float valf1, float valf2, float valf3, float valf4, float valf5, float valf6, float valf7){
	int shiftamount;
	int i=0;
	int insertLoc;
	char opcode[3];
	char retstring[128];
	AI_QuagentsOpCode(cmd,opcode);

	if(bs->currentcommand < bs->finalcommand) now='n';
	if(now=='n'){ //insert at top of order queue, change current order
		if(bs->currentcommand>=QUAGENTSTORECOMMANDSIZE-1){
			if(bs->finalcommand==0){
				//command queue is full.
				Com_sprintf(retstring,127,"er %s %d queuefull\n",opcode,idnum);
				trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
				Com_sprintf(retstring,127,"cp %s %d queuefull\n",opcode,idnum);
				trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
				return;
			}
			//shift command stack
			shiftamount=(bs->finalcommand+1)/2;
			for(i=bs->finalcommand;i<=bs->currentcommand;i++)
				bs->commandqueue[i-shiftamount]=bs->commandqueue[i];
			bs->finalcommand=bs->finalcommand-shiftamount;
			bs->currentcommand=bs->currentcommand-shiftamount;
		}
		bs->currentcommand++;
		insertLoc=bs->currentcommand;
	}else if(now=='r'){
		AIExit_Quagents(bs,bs->commandqueue[bs->currentcommand].id,bs->commandqueue[bs->currentcommand].command,"replaced");
		bs->currentcommand++;
		insertLoc=bs->currentcommand;
	}else{ //insert at bottom of order queue
		if(bs->finalcommand==0){
			if(bs->currentcommand==QUAGENTSTORECOMMANDSIZE-1){
				//command queue is full.
				Com_sprintf(retstring,127,"er %s %d queuefull\n",opcode,idnum);
				trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
				Com_sprintf(retstring,127,"cp %s %d queuefull\n",opcode,idnum);
				trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
				return;
			}
			//shift command stack
			shiftamount=(QUAGENTSTORECOMMANDSIZE-bs->currentcommand+1)/2;
			for(i=bs->currentcommand;i>=bs->finalcommand;i--)
				bs->commandqueue[i+shiftamount]=bs->commandqueue[i];
			bs->finalcommand=bs->finalcommand+shiftamount;
			bs->currentcommand=bs->currentcommand+shiftamount;
		}
		bs->finalcommand--;
		insertLoc=bs->finalcommand;
	}
	bs->commandqueue[insertLoc].command=cmd;
	bs->commandqueue[insertLoc].id=idnum;
	bs->commandqueue[insertLoc].val1=valc1;
	bs->commandqueue[insertLoc].valf1=valf1;
	bs->commandqueue[insertLoc].valf2=valf2;
	bs->commandqueue[insertLoc].valf3=valf3;
	bs->commandqueue[insertLoc].valf4=valf4;
	bs->commandqueue[insertLoc].valf5=valf5;
	bs->commandqueue[insertLoc].valf6=valf6;
	bs->commandqueue[insertLoc].valf7=valf7;
/*	G_Printf("==BEGIN COMMAND STACK DUMP==\n");
	G_Printf("first: %d, final: %d\n",bs->currentcommand,bs->finalcommand);
	for(i=bs->currentcommand;i>=bs->finalcommand;i--){
		G_Printf("%c",bs->commandqueue[i].command);
	}
	G_Printf("\n===END COMMAND STACK DUMP===\n");*/
//	BotRecordNodeSwitch(bs, " Quagent3cmd", "", "gcmd");
}

/*
==================
AINode_GWalkForever
==================
*/
int AINode_GWalkForever(bot_state_t *bs) {
	int movetype;
	vec3_t va;
	bot_moveresult_t moveresult;
	quagent_command_t* cmd;
	cmd=bs->commandqueue+bs->currentcommand;
	va[YAW]=bs->viewangles[YAW]+cmd->valf1;
	va[PITCH]=bs->viewangles[PITCH];
	va[ROLL]=bs->viewangles[ROLL];

	if(bs->lastmovemode==2){
		AIExit_Quagents(bs,cmd->id,cmd->command,"blocked");
		return qfalse;
	}

	AngleVectors(va,va,NULL,NULL);

	memset(&moveresult, 0, sizeof(bot_moveresult_t));
	BotSetupForMovement(bs);
	if(!bs->crouchmode) movetype = MOVE_WALK;
	else movetype = MOVE_CROUCH;
	trap_BotMoveStupidlyInDirection(bs->ms, va, 400*cmd->valf2, movetype,cmd->val1);
	return qtrue;
}

/*
==================
AINode_GWalkTo
==================
*/
int AINode_GWalkTo(bot_state_t *bs) {
	int movetype;
	gentity_t* ent=&g_entities[bs->entitynum];
	vec3_t va;
	vec3_t mins;
	vec3_t maxs;
	bot_moveresult_t moveresult;
	quagent_command_t* cmd;
	trace_t trace;
	int passent;
	float dist;

	cmd=bs->commandqueue+bs->currentcommand;

	if(bs->lastmovemode==2){
		AIExit_Quagents(bs,cmd->id,cmd->command,"blocked");
		return qfalse;
	}

	va[0]=cmd->valf2-ent->client->ps.origin[0];
	va[1]=cmd->valf3-ent->client->ps.origin[1];
	va[2]=0;
	dist=VectorNormalize(va);

	if(dist>20){
		//we're far enough away that we should use the normal walk functionality.
		memset(&moveresult, 0, sizeof(bot_moveresult_t));
		BotSetupForMovement(bs);
		if(!bs->crouchmode) movetype = MOVE_WALK;
		else movetype = MOVE_CROUCH;
		trap_BotMoveStupidlyInDirection(bs->ms, va, 400*cmd->valf1, movetype,cmd->val1);
	}else if(dist==0){
		//we're there
		AIExit_Quagents(bs,cmd->id,cmd->command,"done");
		return qfalse;	
	}else{
		//attempt to snap bot to point.
		va[0]=cmd->valf2;
		va[1]=cmd->valf3;
		va[2]=ent->client->ps.origin[2];

		mins[0]=-16;
		mins[1]=-16;
		mins[2]=-24;
		maxs[0]=16;
		maxs[1]=16;
		maxs[2]=36;

		passent = ent->s.number;

		trap_Trace (&trace, ent->client->ps.origin, mins, maxs, va, passent, MASK_SHOT );
		if(trace.fraction!=1){
			AIExit_Quagents(bs,cmd->id,cmd->command,"blocked");
			return qfalse;
		}

		if(!trap_AAS_PointContents(va)){
			ent->client->ps.eFlags ^= EF_TELEPORT_BIT;
			ent->client->ps.origin[0]=cmd->valf2;
			ent->client->ps.origin[1]=cmd->valf3;
			ent->s.pos.trDelta[0]=0;
			ent->s.pos.trDelta[1]=0;
			ent->client->ps.velocity[0]=0;
			ent->client->ps.velocity[1]=0;
		}else{
			AIExit_Quagents(bs,cmd->id,cmd->command,"blocked");
			return qfalse;
		}
	}
	return qtrue;
}

/*
==================
AINode_GWalkBy
==================
*/
int AINode_GWalkBy(bot_state_t *bs) {
	gentity_t* ent=&g_entities[bs->entitynum];
	int movetype;
	vec3_t va;
	vec3_t mins;
	vec3_t maxs;
	vec3_t point;
	bot_moveresult_t moveresult;
	trace_t trace;
	quagent_command_t* cmd;
	cmd=bs->commandqueue+bs->currentcommand;

	if(bs->lastmovemode==2){
		AIExit_Quagents(bs,cmd->id,cmd->command,"blocked");
		return qfalse;
	}

	if(cmd->valf3 >0){
		VectorCopy(ent->client->ps.origin,bs->moveby_old_origin);
		bs->movebyptr=&(cmd->valf3);

		va[YAW]=bs->viewangles[YAW]+cmd->valf1;
		va[PITCH]=bs->viewangles[PITCH];
		va[ROLL]=bs->viewangles[ROLL];
		AngleVectors(va,va,NULL,NULL);
		memset(&moveresult, 0, sizeof(bot_moveresult_t));
		BotSetupForMovement(bs);
		if(!bs->crouchmode) movetype = MOVE_WALK;
		else movetype = MOVE_CROUCH;

		trap_BotMoveStupidlyInDirection(bs->ms, va, 400*cmd->valf2, movetype,cmd->val1);
	}else{
		//overmoved, snap back.
		mins[0]=-15;
		mins[1]=-15;
		mins[2]=-24;
		maxs[0]=15;
		maxs[1]=15;
		maxs[2]=36;

		point[0] = ent->client->ps.origin[0];
		point[1] = ent->client->ps.origin[1];
		point[2] = ent->client->ps.origin[2] - 0.25;
		trap_Trace(&trace, ent->client->ps.origin, mins, maxs, point, bs->entitynum, MASK_PLAYERSOLID); //use MASK_PLAYERSOLD since we can stand on entities.
		if(trace.fraction<1){ //only teleport if we're on the ground. Otherwise this will wreck jump pads.
			VectorSubtract(ent->client->ps.origin,bs->moveby_old_origin,va); //va is now the distance moved since last frame
			VectorNormalize(va); //va is now the direction moved since last frame
			VectorScale(va,cmd->valf3,va); //va is now the amount to move back

			VectorAdd(ent->client->ps.origin,va,ent->client->ps.origin); //teleported client back.
			ent->client->ps.eFlags ^= EF_TELEPORT_BIT;
			ent->s.pos.trDelta[0]=0;
			ent->s.pos.trDelta[1]=0;
			ent->client->ps.velocity[0]=0;
			ent->client->ps.velocity[1]=0;
			cmd->val1=3;
			bs->movebyptr=NULL;
			AIExit_Quagents(bs,cmd->id,cmd->command,"done");
			return qtrue;
		}else{
			bs->movebyptr=NULL;
			AIExit_Quagents(bs,cmd->id,cmd->command,"overshot");
			return qfalse;
		}
	}
	return qtrue;
}

/*
==================
AINode_GJumpOnce
==================
*/

	// val1=0: jump not begun
	// val1=1: jump began
	// val1=2: jump finished

int AINode_GJumpOnce(bot_state_t *bs) {
	int movetype;
	gentity_t* ent=&g_entities[bs->entitynum];
	vec3_t va, point, mins, maxs;
	trace_t trace;
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	bot_moveresult_t moveresult;

	va[YAW]=bs->viewangles[YAW]+cmd->valf1;
	va[PITCH]=bs->viewangles[PITCH];
	va[ROLL]=bs->viewangles[ROLL];

	AngleVectors(va,va,NULL,NULL);

	mins[0]=-15;
	mins[1]=-15;
	mins[2]=-24;
	maxs[0]=15;
	maxs[1]=15;
	maxs[2]=36;

	point[0] = ent->client->ps.origin[0];
	point[1] = ent->client->ps.origin[1];
	point[2] = ent->client->ps.origin[2] - 0.25;
	trap_Trace(&trace, ent->client->ps.origin, mins, maxs, point, bs->entitynum, MASK_PLAYERSOLID); //use MASK_PLAYERSOLID since we can stand on entities.



	if(trace.fraction==1){ //in the air.
		if(cmd->val1!=0){ //if val1==0, we haven't jumped yet: we must still be falling from some previous movement! Wait until we hit the ground then jump.
			//air control.
			cmd->val1=2;
			memset(&moveresult, 0, sizeof(bot_moveresult_t));
			BotSetupForMovement(bs);
			if(!bs->crouchmode) movetype = MOVE_WALK;
			else movetype = MOVE_CROUCH;
			trap_BotMoveStupidlyInDirection(bs->ms, va, 400*cmd->valf2, movetype,0);
		}
	}else{ //on the ground.
		if(cmd->val1==0) cmd->val1=1;
		if(cmd->val1==1){ //haven't jumped yet, might be in jump cooldown period, keep trying to jump
			memset(&moveresult, 0, sizeof(bot_moveresult_t));
			BotSetupForMovement(bs);
			movetype = MOVE_JUMP;
			trap_BotMoveStupidlyInDirection(bs->ms, va, 400*cmd->valf2, movetype,0);
		}else if(cmd->val1==2){ //we have landed.
			AIExit_Quagents(bs,cmd->id,cmd->command,"done");
			return qfalse;
		}
	}
	return qtrue;
}


/*
 ==================
 AINode_GRotate
 ==================
 */
#define QUAGENT_ROTATE_SPEED 70 //could use built-in stuff, but it's buggy, so we'll just roll our own
int AINode_GRotate(bot_state_t *bs){
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;

	float delta_y=cmd->valf1;
	float delta_p=cmd->valf2;
	float maxrotspeed=QUAGENT_ROTATE_SPEED*bs->thinktime;

	if(delta_y == 0 && delta_p == 0){
		AIExit_Quagents(bs,cmd->id,cmd->command,"done");
		return qfalse;
	}

	if(delta_y > maxrotspeed)       delta_y=maxrotspeed;
	else if(delta_y < -maxrotspeed) delta_y=-maxrotspeed;
	if(delta_p > maxrotspeed)       delta_p=maxrotspeed;
	else if(delta_p < -maxrotspeed) delta_p=-maxrotspeed;
	bs->viewangles[YAW]+=delta_y;
	bs->viewangles[PITCH]+=delta_p;
	cmd->valf1-=delta_y;
	cmd->valf2-=delta_p;
	bs->viewangles[YAW]=AngleModPrecise(bs->viewangles[YAW]);
	bs->viewangles[PITCH]=AngleModPrecise(bs->viewangles[PITCH]);

	if(bs->viewangles[PITCH]>87 && bs->viewangles[PITCH]<180 && delta_p > 0){ //clamp pitches
		bs->viewangles[PITCH]=87;
		cmd->valf2=0;
	}else if(bs->viewangles[PITCH]<273 && bs->viewangles[PITCH]>180 && delta_p < 0){
		bs->viewangles[PITCH]=273;
		cmd->valf2=0;
	}

	VectorCopy(bs->viewangles,bs->ideal_viewangles);
	trap_EA_View(bs->client, bs->viewangles);
	return qtrue;
}


/*
 ==================
 AINode_GPop
 ==================
 */
int AINode_GPop(bot_state_t *bs){
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	cmd=bs->commandqueue+bs->currentcommand;
	AIExit_Quagents(bs,cmd->id,cmd->command,"popped");
	return qfalse;
}


/*
 ==================
 AINode_GForgetAllTasks
 ==================
 */
int AINode_GForgetAllTasks(bot_state_t *bs){
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	while(bs->currentcommand>=bs->finalcommand){
		cmd=bs->commandqueue+bs->currentcommand;
		AIExit_Quagents(bs,cmd->id,cmd->command,"forgotten");
	}
	return qtrue;
}



/*
 ==================
 AINode_GForgetMostTasks
 ==================
 */
int AINode_GForgetMostTasks(bot_state_t *bs){
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	while(bs->currentcommand>bs->finalcommand){
		cmd=bs->commandqueue+bs->currentcommand;
		AIExit_Quagents(bs,cmd->id,cmd->command,"forgotten");
	}
	return qtrue;
}


/*
 ==================
 AINode_GPause
 ==================
 */
int AINode_GPause(bot_state_t *bs){
	return qtrue;
}

/*
 ==================
 AINode_GBasicQuery
 ==================
 */
int AINode_GBasicQuery(bot_state_t *bs){
	char opcode[3];
	char retstring[128];
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	playerState_t ps=g_entities[bs->entitynum].client->ps;
	int qcommand=cmd->id;
	AI_QuagentsOpCode(cmd->command,opcode);
	
	switch(cmd->command){
		case QCMD_HC:
			Com_sprintf(retstring,127,"rs %s %d %d\n",opcode,qcommand,bs->inventory[INVENTORY_HEALTH]);
			trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
			break;
		case QCMD_HM:
			Com_sprintf(retstring,127,"rs %s %d %d\n",opcode,qcommand,bs->cur_ps.stats[STAT_MAX_HEALTH]);
			trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
			break;
		case QCMD_AC:
			Com_sprintf(retstring,127,"rs %s %d %d\n",opcode,qcommand,bs->inventory[INVENTORY_ARMOR]);
			trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
			break;
		case QCMD_AM:
			Com_sprintf(retstring,127,"rs %s %d %d\n",opcode,qcommand,bs->cur_ps.stats[STAT_MAX_HEALTH]); //max health and max armor are the same thing - max_armor does not exist.
			trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
			break;
		case QCMD_LC:
			Com_sprintf(retstring,127,"rs %s %d %lf %lf %lf\n",opcode,qcommand,ps.origin[0],ps.origin[1],ps.origin[2]);
			trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
			break;
		case QCMD_FC:
			Com_sprintf(retstring,127,"rs %s %d %lf %lf %lf\n",opcode,qcommand,bs->viewangles[YAW],bs->viewangles[PITCH],bs->viewangles[ROLL]);
			trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
			break;
		
	}
	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	return qfalse;
}

/*
=================
AINode_GRangeFinder
=================
*/
int AINode_GRangeFinder(bot_state_t *bs) {
	char opcode[3];
	char retstring[128];
	vec3_t		end;
	vec3_t		muzzle;
	vec3_t		forward;
	vec3_t		up;
	vec3_t		right;
	vec3_t		angles;
	vec3_t		maxs;
	vec3_t		mins;
	trace_t		trace;
	int			passent;
	int			dist;
//	gentity_t	*tent;
	gentity_t*	ent=&g_entities[bs->entitynum];
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	dist=(int)(cmd->valf1);

	VectorCopy(ent->client->ps.viewangles,angles);
	angles[YAW]+=cmd->valf2;
	angles[PITCH]+=cmd->valf3;
	AngleVectors (angles, forward, right, up);
	muzzle[0]=ent->client->ps.origin[0];
	muzzle[1]=ent->client->ps.origin[1];
	muzzle[2]=ent->client->ps.origin[2]+30;
	VectorMA (muzzle, dist, forward, end);
	AI_QuagentsOpCode(cmd->command,opcode);
	passent = ent->s.number;

	mins[0]=-15;
	mins[1]=-15;
	mins[2]=-54; //use a skewed box, since rangefinder starts higher than origin.
	maxs[0]=15;
	maxs[1]=15;
	maxs[2]=6;

	switch(cmd->val1){
		case 0: trap_Trace (&trace, muzzle, NULL, NULL, end, passent, MASK_DEADSOLID ); break;
		case 1: trap_Trace (&trace, muzzle, NULL, NULL, end, passent, MASK_PLAYERSOLID  ); break;
		case 2: trap_Trace (&trace, muzzle, NULL, NULL, end, passent, MASK_SHOT   ); break;
		case 3: trap_Trace (&trace, muzzle, NULL, NULL, end, passent, MASK_ALL   ); break;
		case 4: trap_Trace (&trace, muzzle, mins, maxs, end, passent, MASK_DEADSOLID ); break;
		case 5: trap_Trace (&trace, muzzle, mins, maxs, end, passent, MASK_PLAYERSOLID  ); break;
		case 6: trap_Trace (&trace, muzzle, mins, maxs, end, passent, MASK_SHOT   ); break;
		case 7: trap_Trace (&trace, muzzle, mins, maxs, end, passent, MASK_ALL   ); break;
	}
	if(cmd->val1==0) trap_Trace (&trace, muzzle, NULL, NULL, end, passent, MASK_SOLID );
	else if(cmd->val1==1) trap_Trace (&trace, muzzle, NULL, NULL, end, passent, MASK_ALL );
	if(cmd->val1==2) trap_Trace (&trace, muzzle, mins, maxs, end, passent, MASK_SOLID );
	if(cmd->val1==3) trap_Trace (&trace, muzzle, mins, maxs, end, passent, MASK_ALL );
	if(trace.fraction==1){
		Com_sprintf(retstring,127,"rs %s %d %d -1\n",opcode,cmd->id,dist);
		trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
	}else if ( trace.entityNum >= ENTITYNUM_MAX_NORMAL ) {
		Com_sprintf(retstring,127,"rs %s %d %lf -2\n",opcode,cmd->id,dist*trace.fraction);
		trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
	}else{
		Com_sprintf(retstring,127,"rs %s %d %lf %d\n",opcode,cmd->id,dist*trace.fraction,trace.entityNum);
		trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
	}
	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	return qfalse;
}

/*
=================
AINode_GCanSee
=================
*/
int AINode_GCanSee(bot_state_t *bs) {
	char opcode[3];
	char retstring[128];
	vec3_t		start;
	vec3_t		end;
	vec3_t		delta;
	trace_t		trace;
	int			passent;
	float dist;
	float theta;
	float phi=0;
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	gentity_t*	ent=&g_entities[bs->entitynum];
    gentity_t*  target=&g_entities[(int)(cmd->val1)];

	AI_QuagentsOpCode(cmd->command,opcode);

	VectorCopy(ent->client->ps.origin,start);
	start[2]+=30;
	VectorCopy(target->r.currentOrigin,end);
	end[2]+=30;
	VectorSubtract(end,start,delta);
	passent = ent->s.number;
	dist=VectorNormalize(delta);
	theta=atan2(delta[1],delta[0])/3.141592654*180-bs->viewangles[YAW];
	phi=atan2(-delta[2],sqrt(delta[0]*delta[0]+delta[1]*delta[1]))/3.141592654*180-bs->viewangles[PITCH];

	trap_Trace (&trace, start, NULL, NULL, end, passent, MASK_SOLID );
	if(trace.fraction==1 || trace.entityNum==target->s.number){
		Com_sprintf(retstring,127,"rs %s %d t %lf %lf %lf\n",opcode,cmd->id,dist,theta,phi);
		trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
	}else if ( trace.entityNum >= ENTITYNUM_MAX_NORMAL ) {
		Com_sprintf(retstring,127,"rs %s %d f 0 0 0\n",opcode,cmd->id);
		trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
	}
	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	return qfalse;
}


/*
 ==================
 AINode_GPeek
 ==================
 */
int AINode_GPeek(bot_state_t *bs){
	char opcode[3];
	char copcode[3];
	char retstring[128];
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	int qcommand;
	int cmdptr=bs->currentcommand;
	int s=bs->currentcommand-bs->finalcommand;
	AI_QuagentsOpCode(cmd->command,opcode);
	cmdptr--;
    cmdptr-=cmd->val1;
	if(bs->finalcommand>cmdptr){
		Com_sprintf(retstring,127,"rs %s %d %d 00 0\n",opcode,cmd->id,s);
	}else{
		AI_QuagentsOpCode(bs->commandqueue[cmdptr].command,copcode);
		qcommand=bs->commandqueue[cmdptr].id;
		Com_sprintf(retstring,127,"rs %s %d %d %s %d\n",opcode,cmd->id,s,copcode,qcommand);
	}
	trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);

	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	return qfalse;
}


/*
 ==================
 AINode_GShoot
 ==================
 */
int AINode_GShoot(bot_state_t *bs){
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	weaponinfo_t wi;
	int ammoval=0;
	trap_BotGetWeaponInfo(bs->ws, bs->weaponnum, &wi);
	//if fire has to be released to activate weapon
	if (wi.flags & WFL_FIRERELEASED) {
		if (bs->flags & BFL_ATTACKED) {
			trap_EA_Attack(bs->client);
		}
	}
	else {
		trap_EA_Attack(bs->client);
	}
	bs->flags ^= BFL_ATTACKED;
	switch(bs->weaponnum){
		case WP_MACHINEGUN:       ammoval=INVENTORY_BULLETS;       break;
		case WP_SHOTGUN:          ammoval=INVENTORY_SHELLS;        break;
		case WP_GRENADE_LAUNCHER: ammoval=INVENTORY_GRENADES;      break;
		case WP_ROCKET_LAUNCHER:  ammoval=INVENTORY_ROCKETS;       break;
		case WP_LIGHTNING:        ammoval=INVENTORY_LIGHTNINGAMMO; break;
		case WP_RAILGUN:          ammoval=INVENTORY_SLUGS;         break;
		case WP_PLASMAGUN:        ammoval=INVENTORY_CELLS;         break;
		case WP_BFG:              ammoval=INVENTORY_BFGAMMO;       break;
	}
	if(ammoval!=0 && bs->inventory[ammoval]==0){
		AIExit_Quagents(bs,cmd->id,cmd->command,"empty");
	}else{
		AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	}
	return qfalse;
}


/*
 ==================
 AINode_GSelectWeapon
 ==================
 */
int AINode_GSelectWeapon(bot_state_t *bs){
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	char opcode[3];
	char retstring[128];
	int ammoval=0;
	AI_QuagentsOpCode(cmd->command,opcode);

	switch(cmd->val1){
		case 1: if(bs->inventory[INVENTORY_GAUNTLET]) bs->weaponnum=WP_GAUNTLET; else ammoval=-1; break;
		case 2: if(bs->inventory[INVENTORY_MACHINEGUN]) bs->weaponnum=WP_MACHINEGUN; else ammoval=-1; break;
		case 3: if(bs->inventory[INVENTORY_SHOTGUN]) bs->weaponnum=WP_SHOTGUN; else ammoval=-1; break;
		case 4: if(bs->inventory[INVENTORY_GRENADELAUNCHER]) bs->weaponnum=WP_GRENADE_LAUNCHER; else ammoval=-1; break;
		case 5: if(bs->inventory[INVENTORY_ROCKETLAUNCHER]) bs->weaponnum=WP_ROCKET_LAUNCHER; else ammoval=-1; break;
		case 6: if(bs->inventory[INVENTORY_LIGHTNING]) bs->weaponnum=WP_LIGHTNING; else ammoval=-1; break;
		case 7: if(bs->inventory[INVENTORY_RAILGUN]) bs->weaponnum=WP_RAILGUN; else ammoval=-1; break;
		case 8: if(bs->inventory[INVENTORY_PLASMAGUN]) bs->weaponnum=WP_PLASMAGUN; else ammoval=-1; break;
		case 9: if(bs->inventory[INVENTORY_BFG10K]) bs->weaponnum=WP_BFG; else ammoval=-1; break;
		default: ammoval=-1; break;
	}
	if(ammoval==-1){
		AIExit_Quagents(bs,cmd->id,cmd->command,"invalid_weapon");
	}else{
		switch(bs->weaponnum){
			case WP_MACHINEGUN:       ammoval=INVENTORY_BULLETS;       break;
			case WP_SHOTGUN:          ammoval=INVENTORY_SHELLS;        break;
			case WP_GRENADE_LAUNCHER: ammoval=INVENTORY_GRENADES;      break;
			case WP_ROCKET_LAUNCHER:  ammoval=INVENTORY_ROCKETS;       break;
			case WP_LIGHTNING:        ammoval=INVENTORY_LIGHTNINGAMMO; break;
			case WP_RAILGUN:          ammoval=INVENTORY_SLUGS;         break;
			case WP_PLASMAGUN:        ammoval=INVENTORY_CELLS;         break;
			case WP_BFG:              ammoval=INVENTORY_BFGAMMO;       break;
			default:                  ammoval=0;                       break;
		}
		trap_EA_SelectWeapon(bs->client,bs->weaponnum);
		if(ammoval!=0){
			Com_sprintf(retstring,127,"rs %s %d %d\n",opcode,cmd->id,bs->inventory[ammoval]);
			trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
		}
		AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	}
	return qfalse;
}

/*
 ==================
 AINode_GGetAmmo
 ==================
 */
int AINode_GGetAmmo(bot_state_t *bs){
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	char opcode[3];
	char retstring[128];
	int ammoval=0;
	AI_QuagentsOpCode(cmd->command,opcode);

	switch(cmd->val1){
		case 2: ammoval=INVENTORY_BULLETS;       break;
		case 3: ammoval=INVENTORY_SHELLS;        break;
		case 4: ammoval=INVENTORY_GRENADES;      break;
		case 5: ammoval=INVENTORY_ROCKETS;       break;
		case 6: ammoval=INVENTORY_LIGHTNINGAMMO; break;
		case 7: ammoval=INVENTORY_SLUGS;         break;
		case 8: ammoval=INVENTORY_CELLS;         break;
		case 9: ammoval=INVENTORY_BFGAMMO;       break;
		default: ammoval=-1;                     break;
	}
	if(ammoval==-1){
		AIExit_Quagents(bs,cmd->id,cmd->command,"invalid_weapon");
	}else{
		Com_sprintf(retstring,127,"rs %s %d %d\n",opcode,cmd->id,bs->inventory[ammoval]);
		trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
		AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	}
	return qfalse;
}




/*
 ==================
 AINode_GSkip
 ==================
 */
int AINode_GSkip(bot_state_t *bs){
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	return qtrue;
}

/*
 ==================
 AINode_GRadar
 ==================
 */
int AINode_GRadar(bot_state_t *bs){
	char opcode[3];
	char retstring[128];
	int i=0;
	int passent;
	vec3_t start;
	vec3_t end;
	vec3_t delta;
	float dist;
	float theta;
	float phi;
	trace_t trace;
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	gentity_t *consider;
	gentity_t*	ent=&g_entities[bs->entitynum];
	AI_QuagentsOpCode(cmd->command,opcode);

	VectorCopy(ent->client->ps.origin,start);
	start[2]+=30;
	for(i=0;i<level.num_entities;i++){
		consider=&g_entities[i];
		if(consider->inuse){
			VectorCopy(consider->r.currentOrigin,end);
			end[2]+=30;

			VectorSubtract(end,start,delta);
			dist=VectorNormalize(delta);
			if(cmd->valf1>0){ //don't return out-of-range entities (if a positive range was specified)
				if(dist>cmd->valf1) continue;
			}
			if(consider==ent) continue; //don't return self

			passent = ent->s.number;
			trap_Trace (&trace, start, NULL, NULL, end, passent, MASK_SOLID );
			if(trace.fraction < 1) continue; //don't return things the entity can't see.

			theta=atan2(delta[1],delta[0])/3.141592654*180-bs->viewangles[YAW];
			phi=atan2(-delta[2],sqrt(delta[0]*delta[0]+delta[1]*delta[1]))/3.141592654*180-bs->viewangles[PITCH];

			//otherwise, it's on the radar (report entity and result of WhatIs, CanSee).
			Com_sprintf(retstring,127,"rs %s %d %d %s %f %f %f\n",opcode,cmd->id,i,consider->classname,dist,theta,phi);
			trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
		}
	}

	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	return qtrue;
}

/*
 ==================
 AINode_GWhatIs
 ==================
 */
int AINode_GWhatIs(bot_state_t *bs){
	char opcode[3];
	char retstring[128];
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	gentity_t* it;
	int targetid=(int)cmd->valf1;
	AI_QuagentsOpCode(cmd->command,opcode);

	if(targetid==-2){ //wall/floor/ceiling, by our convention in rangefinder.
		Com_sprintf(retstring,127,"rs %s %d wall\n",opcode,cmd->id);
		trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
	}else if(targetid >= 0 && targetid<level.num_entities){
		it=&g_entities[targetid];
		if(it->inuse){
			Com_sprintf(retstring,127,"rs %s %d %s\n",opcode,cmd->id,it->classname);
			trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
		}else{
			Com_sprintf(retstring,127,"rs %s %d nothing\n",opcode,cmd->id);
			trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
		}
	}else{
		Com_sprintf(retstring,127,"rs %s %d nothing\n",opcode,cmd->id);
		trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
	}
	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	return qtrue;
}

/*
==================
AINode_GWalkFor
==================
*/
int AINode_GWalkFor(bot_state_t *bs) {
	int movetype;
	vec3_t va;
	bot_moveresult_t moveresult;
	quagent_command_t* cmd;

	cmd=bs->commandqueue+bs->currentcommand;

	if(bs->lastmovemode==2){
		AIExit_Quagents(bs,cmd->id,cmd->command,"blocked");
		return qfalse;
	}

	if(cmd->valf3 > 0){
		bs->movefor_old_time=level.time;
		bs->moveforptr=&(cmd->valf3);
		va[YAW]=bs->viewangles[YAW]+cmd->valf1;
		va[PITCH]=bs->viewangles[PITCH];
		va[ROLL]=bs->viewangles[ROLL];
		AngleVectors(va,va,NULL,NULL);
		memset(&moveresult, 0, sizeof(bot_moveresult_t));
		BotSetupForMovement(bs);
		movetype = MOVE_WALK;
		trap_BotMoveStupidlyInDirection(bs->ms, va, 400*cmd->valf2, movetype,cmd->val1);
	}else{
		AIExit_Quagents(bs,cmd->id,cmd->command,"done");
		return qfalse;
	}
	return qtrue;
}

/*
=================
AINode_GBatchRangeFinder
=================
*/
int AINode_GBatchRangeFinder(bot_state_t *bs) {
	char opcode[3];
	vec3_t		end;
	vec3_t		muzzle;
	vec3_t		forward;
	vec3_t		angles;
	vec3_t		maxs;
	vec3_t		mins;
	trace_t		trace;
	int			passent;
	gentity_t*	ent=&g_entities[bs->entitynum];
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;

	int rangefindermode=cmd->val1;
	int dist=		(int)cmd->valf1;
	float LLY=		cmd->valf2;
	float LLP=		cmd->valf3;
	int width=		(int)cmd->valf4;
	int height=		(int)cmd->valf5;
	float pxwidth=	cmd->valf6;
	float pxheight=	cmd->valf7;
	int mask;
	int socket=bs->quagentsSocketFD;
	int i,j;
	int entity;
	char retstring[50];
	memset(retstring,0,50);

	AI_QuagentsOpCode(cmd->command,opcode);
	Com_sprintf(retstring,50,"rs %s %d %d",opcode,cmd->id,height*width);
	trap_Com_QuagentsWriteStr(socket,retstring);

	switch(rangefindermode & 0x3){
		case 0: mask=MASK_DEADSOLID; break;
		case 1: mask=MASK_PLAYERSOLID; break;
		case 2: mask=MASK_SHOT; break;
		case 3: mask=MASK_ALL; break;
	}

	mins[0]=-15;
	mins[1]=-15;
	mins[2]=-24;
	maxs[0]=15;
	maxs[1]=15;
	maxs[2]=36;

	passent = ent->s.number;
	VectorCopy(ent->client->ps.viewangles,angles);
	angles[YAW]+=LLY;
	angles[PITCH]+=LLP;
	muzzle[0]=ent->client->ps.origin[0];
	muzzle[1]=ent->client->ps.origin[1];
	muzzle[2]=ent->client->ps.origin[2]+30;

	for(i=0;i<height;i++){
		for(j=0;j<width;j++){
			AngleVectors (angles, forward, NULL,NULL);
			VectorMA (muzzle, dist, forward, end);
			if(rangefindermode & 0x4) trap_Trace (&trace, muzzle, mins, maxs, end, passent, mask );
			else trap_Trace (&trace, muzzle, NULL, NULL, end, passent, mask );

			if(trace.fraction==1)
				 entity=-1;
			else if(trace.entityNum >= ENTITYNUM_MAX_NORMAL)
				entity=-2;
			else
				entity=trace.entityNum;
			memset(retstring,0,50);
			Com_sprintf(retstring,50," %f %d",dist*trace.fraction,entity);
			trap_Com_QuagentsWriteStr(socket,retstring);
			angles[YAW]+=pxwidth;
		}
		angles[PITCH]+=pxheight;
		angles[YAW]=ent->client->ps.viewangles[YAW]+LLY;
	}
	trap_Com_QuagentsWriteStr(socket,"\n");
	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	return qfalse;
}


/*
 ==================
 AINode_GCrouch
 ==================
 */
int AINode_GCrouch(bot_state_t *bs){
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	bs->crouchmode=cmd->val1;
	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	return qfalse;
}

/*
=================
AINode_GShove
=================
*/
int AINode_GShove(bot_state_t *bs) {
	vec3_t		end;
	vec3_t		shovevec;
	vec3_t		forward;
	vec3_t		angles;
	vec3_t		maxs;
	vec3_t		mins;
	trace_t		trace;
	int			dist=30;
	gentity_t	*target;
	gentity_t*	ent=&g_entities[bs->entitynum];
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;

	//do the trace
	VectorCopy(ent->client->ps.viewangles,angles);
	AngleVectors (angles, forward, NULL, NULL);
	mins[0]=-15;
	mins[1]=-15;
	mins[2]=-24;
	maxs[0]=15;
	maxs[1]=15;
	maxs[2]=36;
	VectorMA (ent->client->ps.origin, dist, forward, end);
	trap_Trace(&trace, ent->client->ps.origin, mins, maxs, end, bs->entitynum, MASK_ALL);

	//set up the angle that the shove was at
    angles[YAW]+=cmd->valf2;
    angles[PITCH]+=cmd->valf3;
	AngleVectors (angles, forward, NULL, NULL);
	shovevec[0]=forward[0]*cmd->valf1;
	shovevec[1]=forward[1]*cmd->valf1;
	shovevec[2]=forward[2]*cmd->valf1;

	if(trace.fraction==1){
		AIExit_Quagents(bs,cmd->id,cmd->command,"nothing_there");
	}else if ( trace.entityNum >= ENTITYNUM_MAX_NORMAL ) {
		AIExit_Quagents(bs,cmd->id,cmd->command,"illegal_target");
		return qfalse;
	}else{
		target=&g_entities[trace.entityNum];
		if ( target->client ) {
			VectorAdd (target->s.pos.trDelta, shovevec, target->s.pos.trDelta);
			VectorAdd (target->client->ps.velocity, shovevec, target->client->ps.velocity);
		}else{
			target->s.pos.trType = TR_GRAVITY; //of course this means if you shove a rocket it'll become a grenade... >_<
			target->s.pos.trTime = level.time;
			VectorCopy(target->r.currentOrigin,target->s.pos.trBase);
			VectorCopy(shovevec,target->s.pos.trDelta);
		}
		AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	}
	return qfalse;
}


/*
 ==================
 AINode_GSay
 ==================
 */
int AINode_GSay(bot_state_t *bs){
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	char string[2048];

	char opcode[3];
	char retstring[2048];
	int i=0;
	int passent;
	vec3_t start;
	vec3_t end;
	vec3_t delta;
	float dist;
	trace_t trace;
	bot_state_t* botconsider;
	gentity_t* consider;
	gentity_t* ent=&g_entities[bs->entitynum];
	AI_QuagentsOpCode(cmd->command,opcode);

	//retrieve string from heap
	trap_Com_QuagentsFromHeap(cmd->val1,string);
	trap_Com_QuagentsFreeHeap(cmd->val1);

	//find quagents that can hear.
	VectorCopy(ent->client->ps.origin,start);
	start[2]+=30;
	for (i = 0; i < maxclients && i < MAX_CLIENTS; i++) {
		botconsider=GetBotState(i);
		if ( !botconsider || !botconsider->inuse || botconsider->quagentsSocketFD==-1 || botconsider->quagentsDatastreamSocketFD==-1) continue;
		consider=&g_entities[botconsider->entitynum];
		VectorCopy(consider->r.currentOrigin,end);
		end[2]+=30;

		VectorSubtract(end,start,delta);
		dist=VectorNormalize(delta);
		if(cmd->valf1>=0){ //don't return out-of-range entities (if a positive range was specified)
			if(dist>cmd->valf1) continue;
		}

		passent = ent->s.number;
		trap_Trace (&trace, start, NULL, NULL, end, passent, MASK_SOLID );
		if(trace.fraction < 1) continue; //don't return things the entity can't see.

		//otherwise, it can hear, so write on its socket.
		Com_sprintf(retstring,2047,"ev %s %d %s\n",opcode,bs->entitynum,string);
		trap_Com_QuagentsWriteStr(botconsider->quagentsDatastreamSocketFD,retstring);
	}

	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	return qfalse;
}


/*
 ==================
 AINode_GAlchemize
	: for debugging purposes dealing with inventory and item placement.
	: Having some model loading trouble.
 ==================
 */
int AINode_GAlchemize(bot_state_t *bs){
	quagent_command_t* cmd=bs->commandqueue+bs->currentcommand;
	gentity_t *itement;
	gentity_t *self;
	gitem_t* item;

	char *itemname;
	int itemid = (int)cmd->valf1;
	int quantity = (int)cmd->valf2;
	
	// choose item
	switch(itemid){
		case 0: 
			itemname = "quagent_item_treasure";
			break;	
		case 1: 
			itemname = "quagent_item_gold";
			break;
		default: 
			itemname = "invalid_item";
	}

	if (!strcmp(itemname, "invalid_item"))
		AIExit_Quagents(bs,cmd->id,cmd->command,itemname);
	else if (quantity < 1)
		AIExit_Quagents(bs,cmd->id,cmd->command,"invalid_quantity");
	else {
		vec3_t vel;
		vel[0]=0;
		vel[1]=0;
		vel[2]=200;
		self=&g_entities[bs->entitynum];
		item=BG_FindItemByClassname(itemname);
		item->quantity = quantity;
		itement = G_Spawn();
		itement->s.eType = ET_ITEM;
		itement->s.modelindex = item - bg_itemlist;	// store item number in modelindex
		itement->s.modelindex2 = 1; // This is non-zero if it's a dropped item
		itement->classname = item->classname;
		itement->item = item;
		VectorSet (itement->r.mins, -ITEM_RADIUS, -ITEM_RADIUS, -ITEM_RADIUS);
		VectorSet (itement->r.maxs, ITEM_RADIUS, ITEM_RADIUS, ITEM_RADIUS);
		itement->r.contents = CONTENTS_TRIGGER;
		itement->touch = Touch_Item;
		G_SetOrigin( itement, self->client->ps.origin );
		itement->s.pos.trType = TR_GRAVITY;
		itement->s.pos.trTime = level.time;
		VectorCopy( vel, itement->s.pos.trDelta );
		itement->s.eFlags |= EF_BOUNCE_HALF;
		itement->think = 0;				// don't do anything
		itement->nextthink = 0; 
		itement->flags = FL_DROPPED_ITEM;
		trap_LinkEntity (itement);

		AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	}

	return qfalse;
}

/*
 ==================
 AINode_GPickUp
 ==================
 */

int AINode_GPickUp(bot_state_t *bs) {
	int entityList[MAX_GENTITIES], numEntities, i, pickedUp;
	char retstring[128];
	char opcode[3];
	quagent_command_t* cmd = bs->commandqueue+bs->currentcommand;
	vec3_t	maxs;
	vec3_t	mins;
	gentity_t *self;
	gentity_t *target;
	self = &g_entities[bs->entitynum];
	AI_QuagentsOpCode(cmd->command,opcode);

	// player's bounding box
	mins[0] = self->client->ps.origin[0]-15;
	mins[1] = self->client->ps.origin[1]-15;
	mins[2] = self->client->ps.origin[2]-24;
	maxs[0] = self->client->ps.origin[0]+15;
	maxs[1] = self->client->ps.origin[1]+15;
	maxs[2] = self->client->ps.origin[2]+36;

	// check for entities in bounding box
	numEntities = trap_EntitiesInBox(mins, maxs, entityList, MAX_GENTITIES);

	pickedUp = 0;

	for (i = 0; i < numEntities; i++) {
		target=&g_entities[entityList[i]];
	
		// ignore yourself
		if (target == self)
			continue;
		
		// not a quagent item
		if (target->item->giType != IT_QUAGENTITEM) {
			Com_sprintf(retstring,127,"rs %s %d f %s\n", opcode, cmd->id, target->classname);
			trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
		}
		// pick up quagent item
		else {
			Com_sprintf(retstring,127,"rs %s %d t %d %s\n", opcode, cmd->id, target->item->quantity, target->classname);
			trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
			target->nextthink = level.time + 10;
			target->think = G_FreeEntity;						// free the entity
			bs->inventory[target->item->giTag] += target->item->quantity;		// add to inventory
			pickedUp++;
		}
	}
	
	if (numEntities == 1)
		AIExit_Quagents(bs,cmd->id,cmd->command,"nothing_there");
	else
		AIExit_Quagents(bs,cmd->id,cmd->command,"done");

	return qfalse;
}

/*
 ==================
 AINode_GPutDown
 ==================
 */
int AINode_GPutDown(bot_state_t *bs) {
	quagent_command_t* cmd = bs->commandqueue+bs->currentcommand;
	gentity_t *itement;
	gentity_t *self;
	gitem_t* item;
	char *itemname;
	int itemid = (int)cmd->valf1;
	int quantity = (int)cmd->valf2;
	int index;
	
	// choose item
	switch(itemid){
		case 0: 
			itemname = "quagent_item_treasure";
			index = INVENTORY_QUAGENT_TREASURE;
			break;	
		case 1: 
			itemname = "quagent_item_gold";
			index = INVENTORY_QUAGENT_GOLD;
			break;
		default: 
			itemname = "invalid_item";
			index = -1;
	}

	if (index == -1)
		AIExit_Quagents(bs,cmd->id,cmd->command,itemname);
	else if (!bs->inventory[index])
		AIExit_Quagents(bs,cmd->id,cmd->command,"none_in_inventory");
	else if (quantity < 1)
		AIExit_Quagents(bs,cmd->id,cmd->command,"invalid_quantity");
	else {								// create item
		vec3_t vel;  							
		vel[0]=0;
		vel[1]=0;
		vel[2]=200;
		self=&g_entities[bs->entitynum];
		item=BG_FindItemByClassname(itemname);
		if (quantity > bs->inventory[index])			// if specified quantity was greater than in inventory
			quantity = bs->inventory[index];
		item->quantity = quantity;
		bs->inventory[index] -= quantity;			// subtract from inventory
		itement = G_Spawn();
		itement->s.eType = ET_ITEM;
		itement->s.modelindex = item - bg_itemlist;		// store item number in modelindex
		itement->s.modelindex2 = 1;				// This is non-zero if it's a dropped item
		itement->classname = item->classname;
		itement->item = item;
		VectorSet (itement->r.mins, -ITEM_RADIUS, -ITEM_RADIUS, -ITEM_RADIUS);
		VectorSet (itement->r.maxs, ITEM_RADIUS, ITEM_RADIUS, ITEM_RADIUS);
		itement->r.contents = CONTENTS_TRIGGER;
		itement->touch = Touch_Item;
		G_SetOrigin( itement, self->client->ps.origin );
		itement->s.pos.trType = TR_GRAVITY;
		itement->s.pos.trTime = level.time;
		VectorCopy( vel, itement->s.pos.trDelta );
		itement->s.eFlags |= EF_BOUNCE_HALF;
		itement->think = 0;				// don't do anything
		itement->nextthink = 0; 
		itement->flags = FL_DROPPED_ITEM;
		trap_LinkEntity (itement);

		AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	}

	return qfalse;
}

/*
 ========================
 AINode_GCheckInventory
 ========================
 */
int AINode_GCheckInventory(bot_state_t *bs) {
	quagent_command_t* cmd = bs->commandqueue+bs->currentcommand;
	char *item;
	char retstring[128];
	char opcode[3];
	int i;

	AI_QuagentsOpCode(cmd->command,opcode);
	// loops through quagent items, prints out name and quantity of each item
	for (i = QUAGENT_ITEM_START; i < QUAGENT_ITEM_END; i++) {
		switch(i){
		case INVENTORY_QUAGENT_TREASURE: 
			item = "quagent_item_treasure";
			break;
		case INVENTORY_QUAGENT_GOLD: 
			item = "quagent_item_gold";
			break;
		default: 
			item = "unknown_item";
		}
		Com_sprintf(retstring,127,"rs %s %d %s %d\n", opcode, cmd->id, item, bs->inventory[i]);
		trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);
	}

	AIExit_Quagents(bs,cmd->id,cmd->command,"done");
	return qfalse;
}

/*
 =================
 AINode_GGetView
 =================
  : All bots run on the server, but all rendering runs on the client. This is a major problem
  : when the client and server are not on the same machine or do not have access to the same
  : memory. See the comments in qcommon/common.c:Trap_Com_QuagentsGLGetView.
  : 
  : In architectures where double-buffering is impossible this may disrupt the user's experience.
  : However, I think this is less important than maintaining video accelleration on the draw surface,
  : so nothing further will be done.
 */

int AINode_GGetView(bot_state_t *bs) {
	quagent_command_t* cmd = bs->commandqueue+bs->currentcommand;
	gentity_t* ent=&g_entities[bs->entitynum];
	playerState_t* ps;
	char retstring[128];
	char retstring2[128];
	char opcode[3];

	AI_QuagentsOpCode(cmd->command,opcode);
	Com_sprintf(retstring2,127,"rs %s %d drawing\n", opcode, cmd->id);
	trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring2);

	Com_sprintf(retstring,127,"rs %s %d ", opcode, cmd->id); //trap will handle the rest of the rs.
	trap_Com_QuagentsWriteStr(bs->quagentsSocketFD,retstring);

	ps=&(ent->client->ps);
	if(cmd->command==QCMD_VC) trap_Com_QuagentsGLGetView(bs->quagentsSocketFD,ps,0);
	if(cmd->command==QCMD_VD) trap_Com_QuagentsGLGetView(bs->quagentsSocketFD,ps,1);
	AIExit_Quagents(bs,cmd->id,cmd->command,"done");

	return qfalse;
}

