#!/usr/bin/perl

# Usage: makemap.pl mapname
# run in same directory as the GIF image files for each zoom level

# this script uses the "crop" program 
# you must compile crop and make sure it is in in the
#  same directory as this file

use strict;
use vars qw($crop $db @XNUMS @YNUMS);

$crop = $0;			# get dir this file is in
$crop =~ s/[^\/]*$/crop/;	# replace makemap.pl with crop

if ($#ARGV < 0) {
  print STDERR "$0: Wrong number of arguments\n\n";
  print STDERR "Usage: $0 mapname\n";
  exit(1);
}

my($map) = @ARGV;
my $found = 0;

$db = "$map.db";

open(DB, $db);
while (<DB>) {
  chop;
  next if (/^$/);
  &process($_);
  $found = 1;
}
close(DB);

if (!$found) {
  print "Guessing sizes for panels: $map.db not found\n";

  my $directory = "$map";
  &prepare_dir($directory, $map);
  my($d) = 1;
  $XNUMS[0] = 1; $YNUMS[0] = 1;
  while (-f "$directory-$d.gif") {
    my($width, $height) = &gifsize("$directory-$d.gif");
    $XNUMS[$d] = &max(1, int($width * 2 / 500)/2);
    $YNUMS[$d] = &max(1, int($height * 2 / 350)/2);
    &level($d, $directory, $map);
    $d++;
  }
  $d--;
  print "$map:$map:$d:", join(',', @XNUMS), ":", join(',', @YNUMS), "\n";
  # create file with number of panels and title
  open(DB, ">$map.db");
  print DB "$map:$map:$d:", join(',', @XNUMS), ":", join(',', @YNUMS), "\n";
  close DB;
}

exit;

sub process
{
  my($input) = @_;

  my(undef, $title, $depth, $xnums, $ynums, $tochoose, $defaultfirst) = split(/:/, $input); 
  my $nums = "${xnums}x$ynums";

  print "Making map \"$map\" with ($nums)\n\n";

  my($xnums, $ynums) = split(/x/, $nums);
  @XNUMS = split(/,/, $xnums);
  @YNUMS = split(/,/, $ynums);
  if ($depth != $#XNUMS || $#XNUMS != $#YNUMS) {
    print STDERR "Inconsistent input: depth, x series and y series do not all match\n";
    exit(1);
  }
  
#if (!(($x,$y) = ($size =~ /^(\d+),(\d+)$/))) {
#  print STDERR "$0: Invalid size - should be of form x,y\n";
#  exit(1);
#}
  
  my $directory = "$map";
  &prepare_dir($directory, $map);

  my $level;
  foreach $level (1..$depth) {
    &level($level, $directory, $map);
  }
}
  
###################################################################

sub prepare_dir {
  my($directory, $map) = @_;
  &validate($directory, 0);

  print "Preparing directory...";

  my $files;

  if (-d $directory) {
    $files = unlink(<$directory/$map*gif>);
    printf("%s cleared (%d files removed)\n\n", $directory, $files);
  } else {
    mkdir($directory, 0775);
    print "$directory created\n\n";
  }
  
  print "Linking root map...";
  link("$directory-0.gif", "$directory/$map-0-0-0.gif") || die "Can't link $directory/$map-0-0-0.gif";
  print "root map linked\n\n";
}

###################################################################

sub level 
{
  my($num, $directory, $map) = @_;
  
  my($width, $height) = &gifsize("$directory-$num.gif");

  &validate($directory, $num);
  print "Making depth $num maps...\n";

  my $ox = int($width / $XNUMS[$num]);
  my $oy = int($height / $YNUMS[$num]);

  my($i, $j);
  open(CROP, "|$crop $directory-$num.gif") || die("can't run crop program: $crop");
  foreach $j (0.. ($XNUMS[$num] * 2 - 2)) {
    foreach $i (0..($YNUMS[$num] * 2 - 2)) {
      my $nx = $j / 2;
      my $ny = $i / 2;
      my $dx = int($width / $XNUMS[$num] * $nx);      
      my $dy = int($height / $YNUMS[$num] * $ny);
      print CROP "${ox}x${oy}+${dx}+${dy} $directory/$map-$num-$nx-$ny.gif\n";

#      print "${ox}x${oy}+${dx}+${dy} $directory/$map-$num-$nx-$ny.gif ";
#      print "OK\n";
#      print "${ox}x${oy}+${dx}+${dy} $directory/$map-$num-$nx-$ny.gif\n";
      # above notification now given at appropriate time by crop
      # note: convert silently fails to crop if given decimal geometry args
#    print "  convert -crop ${width}x${height}+${dx}+${dy} $map-$num.gif $map/$map-$num-$n.gif\n";

#    system "convert -crop ${width}x${height}+${dx}+${dy} $map-$num.gif $map/$map-$num-$n.gif";
   }
  }
#  print "CLOSE\n";
  close(CROP);
  print "DONE!\n\n";
}

###################################################################

sub validate {
  my ($directory, $num) = @_;

  if ($num == 0) {
    print "Validating root map...";
  } else {
    print "Validating depth $num map...";
  }

  if (-f "$directory-$num.gif") {
    print "$directory-$num.gif OK\n";
  } else {
    print "$directory-$num.gif not found\n";
    exit(1);
  }
}


sub gifsize 
{
  my($filename) = @_;
  my $header;

  open(GIF, $filename) || die "Gif not found: $filename\n";
  read(GIF, $header, 10);
  my @header = unpack("a6C4", $header);
  if ($header[0] ne 'GIF87a' && $header[0] ne 'GIF89a') {
    printf STDERR "Not a gif file: $filename\n";
  }
  my $width = $header[2] << 8 | $header[1];
  my $height = $header[4] << 8 | $header[3];
  return ($width, $height);
}

sub max {
  local($a, $b) = @_;
  $a > $b ? $a : $b;
}
