/*
** Usage: mapscan [-a] mapfile coordinate
**
**    -a will return all routes at a point, separated by newline characters '\n'
**
** based on: mapper 1.2
**    7/26/93 Kevin Hughes, kevinh@pulua.hcc.hawaii.edu
**    "macmartinized" polygon code copyright 1992 by Eric Haines, erich@eye.com
**
** 11/13/93: Rob McCool, robm@ncsa.uiuc.edu
**
** long time ago: sheikh added 'line' code
**                sheikh added -a option
**
**  8/14/97: sheikh removed "content type" from errors
**                  allows negative coordinates
**                  doesn't require the default
**                  doesn't return MISSED if that's the default
**                  -i only returns a matching inset
**                  no flag skips insets
**
**  4/23/98: gildea not returning MISSED is screwing up select.pl,
**		    removed special case.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#define MAXLINE 500
#define MAXVERTS 100
#define X 0
#define Y 1
#define LINEWIDTH x_coords[0]

#define sqr(x) ((x) * (x))

int isname(char);

int main(int argc, char **argv)
{
  char *mapname, def[MAXLINE];
  double testpoint[2], *pointarray_x, *pointarray_y;
  int maxverts;
  int i, j, k, all, inset, negative;
  FILE *fp;
  char *t;
  
  all = 0;
  inset = 0;

  def[0] = '\0';

  while((argc > 1) && (*(t = argv[1]) == '-')) {
    argc--;
    argv++;
    t++;

    if (*t == 'a')
      all = 1;
    else if (*t == 'i')
      inset = 1;
  }
  /*
    printf("%s %s\n\n", all ? "ALL" : "-", inset ? "INSET" : "-");
    */

  if (argc != 3)
    servererr("Wrong number of arguments.");
  
  mapname = argv[1];
  
  if((!mapname) || (!mapname[0]))
    servererr("No map name given.");
  
  if(!(t = strchr(argv[2],',')))
    servererr("Incorrect coordinates format given.");
  *t++ = '\0';
  testpoint[X] = (double) atoi(argv[2]);
  testpoint[Y] = (double) atoi(t);
  
  if(!(fp=fopen(mapname,"r")))
    servererr("Couldn't open map file.");
  
  maxverts = MAXVERTS;
  pointarray_x = (double*)malloc(sizeof(double)*maxverts);
  pointarray_y = (double*)malloc(sizeof(double)*maxverts);

  while(!feof(fp)) {
    char type[MAXLINE];
    char url[MAXLINE];
    char num[10];
    
    char buf[MAXLINE];
    char *input,*tmp_dbuf,*dbuf_ptr;
    int dbuf_len;
    fgets(buf, MAXLINE, fp);

    // this mess is to keep from having restrictions on the buffer size
    input=buf;
    dbuf_len=MAXLINE;
    while (!feof(fp) && input[strlen(input)-1]!='\n') {
      tmp_dbuf=(char *) malloc(MAXLINE+dbuf_len);
      memcpy(tmp_dbuf,input,dbuf_len);
      dbuf_len += MAXLINE;

      if (input != buf) {
	free(input);
      }
      input=tmp_dbuf;
      dbuf_ptr=input+strlen(input);
      fgets(dbuf_ptr, MAXLINE, fp);
    }
    // end buffer mashing
    
    if((input[0] == '#') || (!input[0]))
      continue;
    
    type[0] = '\0';url[0] = '\0';
    
    for(i=0;isname(input[i]) && (input[i]);i++)
      type[i] = input[i];
    type[i] = '\0';
    
    while(isspace(input[i])) ++i;
    for(j=0;input[i] && isname(input[i]);++i,++j)
      url[j] = input[i];
    url[j] = '\0';
    
    if(!strcmp(type,"default")) {
      strcpy(def,url);
      continue;
    }

    if (!strncmp("INSET=", url, 6) && !all && !inset)
      continue;

    k=0;
    while (input[i]) {
      while (isspace(input[i]) || input[i] == ',')
	i++;
      j = 0;
      negative = 1;
      if (input[i] == '-') {
	negative = -1;
	i++;
      }
      while (isdigit(input[i]))
	num[j++] = input[i++];
      num[j] = '\0';
      if (num[0] != '\0')
	pointarray_x[k] = negative * ((double) atoi(num));
      else
	break;
      while (isspace(input[i]) || input[i] == ',')
	i++;
      j = 0;
      negative = 1;
      if (input[i] == '-') {
        negative = -1;
        i++;
      }
      while (isdigit(input[i]))
	num[j++] = input[i++];
      num[j] = '\0';
      if (num[0] != '\0')
	pointarray_y[k++] = negative * ((double) atoi(num));
      else {
	fclose(fp);
	servererr("Missing y value.");
      }
      /*
	printf("%f,%f\n",pointarray_x[k-1], pointarray_y[k-1]);
	*/
      if (k == maxverts) {
	double *tmp;
	tmp = (double*)malloc(sizeof(double)*maxverts*2);
	memcpy(tmp, pointarray_x, sizeof(double)*maxverts);
	free(pointarray_x);
	pointarray_x = tmp;
	tmp = (double*)malloc(sizeof(double)*maxverts*2);
	memcpy(tmp, pointarray_y, sizeof(double)*maxverts);
	maxverts *= 2;
	free(pointarray_y);
	pointarray_y = tmp;
      }
    }
    if (input != buf) {
      free(input);
    }
    pointarray_x[k] = -1;
    if(!strcmp(type,"poly"))
      if(pointinpoly(testpoint,pointarray_x, pointarray_y))
	sendmesg(url, all);
    if(!strcmp(type,"circle"))
      if(pointincircle(testpoint,pointarray_x, pointarray_y))
	sendmesg(url, all);
    if(!strcmp(type,"rect"))
      if(pointinrect(testpoint,pointarray_x, pointarray_y))
	sendmesg(url, all);
    if(!strcmp(type,"line"))
      if(pointinline(testpoint,pointarray_x, pointarray_y))
	sendmesg(url, all);
  }

  if(def[0])
    sendmesg(def, 0);

  /*
    servererr("No default specified.");
    */
}

sendmesg(char *url, int all)
{
  printf("%s\n", url);

  /*
    printf("Location: %s%c%c",url,10,10);
    printf("This document has moved <A HREF=\"%s\">here</A>%c",url,10); 
    */

  if (!all)
    exit(1);
}

int pointinrect(double point[2], double *x_coords, double *y_coords)
{
        return ((point[X] >= x_coords[0] && point[X] <= x_coords[1]) &&
        (point[Y] >= y_coords[0] && point[Y] <= y_coords[1]));
}

int pointincircle(double point[2], double *x_coords, double *y_coords)
{
        int radius1, radius2;

        radius1 = ((y_coords[0] - y_coords[1]) * (y_coords[0] - y_coords[1]))
	  + ((x_coords[0] - x_coords[1]) * (x_coords[0] - x_coords[1]));
        radius2 = ((y_coords[0] - point[Y]) * (y_coords[0] - point[Y])) +
	  ((x_coords[0] - point[X]) * (x_coords[0] - point[X]));
        return (radius2 <= radius1);
}

int pointinpoly(double point[2], double *x_pgon, double *y_pgon)
{
        int i, numverts, inside_flag, xflag0;
        int crossings;
        double *p, *stop;
        double tx, ty, y;

        for (i = 0; x_pgon[i] != -1; i++)
                ;
        numverts = i;
        crossings = 0;

        tx = point[X];
        ty = point[Y];
        y = y_pgon[numverts - 1];

        p = y_pgon;
        if ((y >= ty) != (*p >= ty)) {
                if ((xflag0 = (x_pgon[numverts - 1] >= tx)) ==
                (x_pgon[0] >= tx)) {
                        if (xflag0)
                                crossings++;
                }
                else {
                        crossings += (x_pgon[numverts - 1] - (y - ty) *
				      (x_pgon[0] - x_pgon[numverts - 1]) /
				      (*p - y)) >= tx;
                }
        }


	for (i = 0; i+1 < numverts; i++) {
		y = y_pgon[i];
                if (y >= ty) {
                        while ((i+1 < numverts) && (y_pgon[i+1] >= ty))
                                i++;
                        if (i+1 >= numverts)
                                break;
                        if ((xflag0 = (x_pgon[i] >= tx)) == (x_pgon[i+1] >= tx)) {
                                if (xflag0)
                                        crossings++;
                        }
                        else {
                                crossings += (x_pgon[i] - (y_pgon[i] - ty) *
					      (x_pgon[i+1] - x_pgon[i]) / (y_pgon[i+1] - y_pgon[i])) >= tx;
                        }
                }
                else {
                        while ((i+1 < numverts) && (y_pgon[i+1] < ty))
                                i++;
                        if (i+1 >= numverts)
                                break;
                        if ((xflag0 = (x_pgon[i] >= tx)) == (x_pgon[i+1] >= tx)) {
                                if (xflag0)
                                        crossings++;
                        }
                        else {
                                crossings += (x_pgon[i] - (y_pgon[i] - ty) *
                                (x_pgon[i+1] - x_pgon[i]) / (y_pgon[i+1] - y_pgon[i])) >= tx;
                        }
                }
        }
        inside_flag = crossings & 0x01;
        return (inside_flag);
}

int pointinline(double point[2], double *x_coords, double *y_coords)
{
  int i;
  float lsqrd, l, r, s, d;

  for (i = 1; x_coords[i+1] != -1; i++) {
    lsqrd = sqr(x_coords[i+1]-x_coords[i]) + sqr(y_coords[i+1]-y_coords[i]);

    if (lsqrd > 0) {
      l = sqrt(lsqrd);
      s = (((y_coords[i]-point[Y]) * (x_coords[i+1]-x_coords[i])) 
	   - ((x_coords[i]-point[X]) * (y_coords[i+1]-y_coords[i]))) / lsqrd;

/*      printf("%f * %f = %f < %f\n", s, l, fabs(s*l), LINEWIDTH); */
      if (fabs(s*l) < LINEWIDTH) {
	r = (((y_coords[i]-point[Y]) * (y_coords[i]-y_coords[i+1])) 
	     - ((x_coords[i]-point[X]) * (x_coords[i+1]-x_coords[i]))) / lsqrd;
	d = LINEWIDTH / l;

/*	printf("%.0f,%.0f - %.0f,%.0f  : l=%f : r=%f : s=%f : d=%f : w=%f\n", x_coords[i], 
		y_coords[i], x_coords[i+1], y_coords[i+1], l, r, s, d, LINEWIDTH);*/

	if ((r > -d) && (r < 1 + d)) {
	  return 1;
	}
      }
    }
  }
  
  return 0;
}

servererr(char *msg)
{
  /*
    printf("Content-type: text/html%c%c",10,10);
    printf("<title>Mapping Server Error</title>");
    printf("<h1>Mapping Server Error</h1>");
    printf("This server encountered an error:<p>");
    */

  printf("ERROR:%s\n", msg);
  exit(-1);
}

int isname(char c)
{
  return (!isspace(c));
}
